<?php

namespace Kozhilya\MybbConnection\Util;

/**
 * Класс, обеспечивающий корректную кодировку при загрузке и отправке данных на форум Mybb.ru
 */
final class MybbEncoder
{
    /**
     * @param string $char
     * @return string
     */
    private static function unicodeEntity(string $char): string
    {
        $char = mb_convert_encoding($char, 'UTF-32', 'UTF-8');
        $t = unpack("N*", $char);
        $t = array_map(function ($n) {
            return sprintf('&#%s;', $n);
        }, $t);

        return implode("", $t);
    }

    /**
     * @param string $string
     * @param int $pointer
     * @return string|null
     */
    private static function nextChar(string $string, int &$pointer): ?string
    {
        if (!isset($string[$pointer])) {
            return null;
        }
        $char = ord($string[$pointer]);
        if ($char < 128) {
            return $string[$pointer++];
        }
        else {
            if ($char < 224) {
                $bytes = 2;
            }
            elseif ($char < 240) {
                $bytes = 3;
            }
            elseif ($char < 248) {
                $bytes = 4;
            }
            elseif ($char == 252) {
                $bytes = 5;
            }
            else {
                $bytes = 6;
            }
            $str = substr($string, $pointer, $bytes);
            $pointer += $bytes;

            return $str;
        }
    }

    /**
     * Преобразование строки UTF-8 в кодировку, совместимую с форумами mybb.ru
     *
     * @param string $input Исходная строка в UTF-8
     * @return string Строка в кодировке форума (Windows-1251)
     */
    public static function toForumEncoding(string $input): string
    {
        $test = iconv('CP1251', 'UTF-8', iconv('UTF-8', 'CP1251//IGNORE', $input));

        $i = 0;
        $j = 0;
        $result = '';

        while (true) {
            $a = MybbEncoder::nextChar($input, $i);
            $b = MybbEncoder::nextChar($test, $j);
            if (is_null($a)) {
                break;
            }
            $dif = '';
            while ($a != $b) {
                $dif .= $a;
                $a = MybbEncoder::nextChar($input, $i);
                if (is_null($a)) {
                    break;
                }
            }
            $result .= MybbEncoder::unicodeEntity($dif);
        }

        return iconv('UTF-8', 'CP1251', $result);
    }

    /**
     * Преобразование строки из кодировки, отправляемой форумами mybb.ru, в UTF-8
     *
     * @param string $input
     * @return string
     */
    public static function fromForumEncoding(string $input): string
    {
        return mb_convert_encoding($input, 'UTF-8', 'CP-1251');
    }
}