<?php

namespace Kozhilya\MybbConnection\FormProcessor;

use Kozhilya\MybbConnection\Exceptions\FormKeyParserException;
use Kozhilya\MybbConnection\Exceptions\MybbCurlException;
use Kozhilya\MybbConnection\Exceptions\MybbFormProcessorMissingOptionsException;
use Kozhilya\MybbConnection\Exceptions\MybbLoginFailedException;
use Kozhilya\MybbConnection\MybbConnection;
use Kozhilya\MybbConnection\MybbLoader;
use Kozhilya\MybbConnection\MybbResponse;
use Kozhilya\MybbConnection\MybbUser;
use Symfony\Component\DomCrawler\Crawler;
use Symfony\Component\DomCrawler\Form;

/**
 * Общий класс-обработчик форм Mybb.ru
 */
abstract class MybbFormProcessor
{
    /**
     * Подключение к Mybb.ru
     * @var MybbConnection
     */
    protected readonly MybbConnection $connection;

    /**
     * Пользователь, выполняющий отправку формы
     * @var MybbUser
     */
    protected readonly MybbUser $user;

    /**
     * Генератор запросов к Mybb.ru
     * @var MybbLoader
     */
    private MybbLoader $loader;

    /**
     * Страница, с которой происходит отправка формы
     * @var string
     */
    public string $referer = '';

    /**
     * Кнопка отправки формы
     * @var string
     */
    public string $button_selector = '[name="submit"]';

    /**
     * Поля формы для отправки
     * @var array
     */
    protected array $fields = [];

    /**
     * Форма для отправки
     * @var Form
     */
    protected Form $form;

    /**
     * @throws MybbCurlException
     * @throws MybbLoginFailedException
     */
    public function __construct(MybbConnection $connection, MybbUser $user)
    {
        $this->connection = $connection;
        $this->user = $user;

        $this->loader = $this->connection->createLoader($user);
        $this->referer = $this->connection->getForumLink();
    }

    /**
     * Инициализация обработчика дополнительными параметрами
     *
     * @param array $options Дополнительные параметры
     *
     * @throws MybbFormProcessorMissingOptionsException
     * @throws FormKeyParserException
     * @throws MybbCurlException
     */
    public abstract function init(array $options): void;

    /**
     * Проверка списка дополнительных параметров
     *
     * @param array $options Дополнительные параметры
     * @param array $keys Список обязательных ключей
     * @return void
     * @throws MybbFormProcessorMissingOptionsException
     */
    protected function assertOptions(array $options, array $keys): void
    {
        $missing = [];

        foreach ($keys as $key) {
            if (!key_exists($key, $options)) {
                $missing[] = $key;
            }
        }

        if (!empty($missing)) {
            throw new MybbFormProcessorMissingOptionsException(
                $missing,
                $keys,
                static::class
            );
        }
    }

    /**
     * Генератор ключей формы
     *
     * @param string $referer Страница, с которой происходит отправка формы
     * @return FormKeyParser
     *
     * @throws FormKeyParserException
     * @throws MybbCurlException
     */
    public function getFormKeys(string $referer): FormKeyParser
    {
        $response = $this->loader->load($referer);
        $html = $response->html;

        $this->referer .= $referer;

        $parser = new FormKeyParser($html);
        $this->extendFields($parser->process());

        $crawler = new Crawler(null, $this->referer);
        $crawler->addHtmlContent(preg_replace('#\s*<script[\s\S]*?</script>\s*#', '', $html));

        $formCrawler = $crawler->filter('#pun-main form');

        $this->form = $formCrawler->filter($this->button_selector)->form();
        $this->extendFields($this->form->getPhpValues());

        return $parser;
    }

    /**
     * Отправка формы
     *
     * @throws MybbCurlException
     */
    public function submit(): MybbResponse
    {
        $url = $this->form->getUri();

        $curl = $this->loader->createCurl($url, [
            CURLOPT_REFERER => $this->referer,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $this->fields,
        ]);

        return $this->loader->exec($curl);
    }

    /**
     * Список полей формы
     *
     * @return array
     */
    public function getFields(): array
    {
        return $this->fields;
    }

    /**
     * Установка поля формы
     *
     * @param string $key
     * @param string $value
     * @return void
     */
    public function setField(string $key, string $value): void
    {
        $this->fields[$key] = $value;
    }

    /**
     * Дополнение полей формы
     *
     * @param array $data
     * @return void
     */
    public function extendFields(array $data): void
    {
        $this->fields = $this->fields + $data;
    }
}