<?php

namespace Kozhilya\MybbConnection;

use Kozhilya\MybbConnection\Delayers\AbstractDelayer;
use Kozhilya\MybbConnection\Exceptions\MybbCurlException;
use Kozhilya\MybbConnection\Exceptions\MybbLoginFailedException;
use Kozhilya\MybbConnection\FormProcessor\MybbFormProcessor;

/**
 * Класс, обеспечивающий подключение к форуму Mybb.ru
 */
abstract class MybbConnection
{
    /**
     * Ссылка на форум, с протоколом и доменом.
     * Пример правильной ссылки на форум: "https://forum.mybb.ru/".
     *
     * @return string
     */
    public abstract function getForumLink(): string;

    /**
     * Построение ссылки на форум
     *
     * @param string $url
     * @return string
     */
    public function buildLink(string $url): string
    {
        if (!str_starts_with($url, 'http')) {
            $forumLink = $this->getForumLink();

            if (!str_ends_with($forumLink, '/')) {
                $forumLink .= '/';
            }

            $url = $forumLink.$url;
        }

        return $url;
    }


    /**
     * Объект для обеспечения запросов к Mybb.ru в очереди
     * @var AbstractDelayer
     */
    private AbstractDelayer $delayer;

    /**
     * Установка нового объекта для обеспечения запросов к Mybb.ru в очереди
     *
     * @return AbstractDelayer
     */
    public function getDelayer(): AbstractDelayer
    {
        return $this->delayer;
    }

    /**
     * Объект для обеспечения запросов к Mybb.ru в очереди
     *
     * @param AbstractDelayer $delayer
     * @return MybbConnection
     */
    public function setDelayer(AbstractDelayer $delayer): MybbConnection
    {
        $this->delayer = $delayer;

        return $this;
    }

    /**
     * Создание генератора запросов к Mybb.ru
     *
     * @throws MybbCurlException
     * @throws MybbLoginFailedException
     */
    public function createLoader(?MybbUser $user): MybbLoader
    {
        return new MybbLoader($this, $user);
    }

    /**
     * Создание нового обработчика форм
     *
     * @template T of MybbFormProcessor
     * @param MybbUser $user Пользователь, отправляющий форму
     * @param class-string<T> $className Имя класса-обработчика формы
     * @param array|null $options Дополнительные настройки обработчика
     *
     * @return T
     */
    public function createFormProcessor(MybbUser $user, string $className, ?array $options = null): MybbFormProcessor
    {
        $processor = new $className($this, $user);

        $processor->init($options ?? []);

        return $processor;
    }

}