<?php

namespace Kozhilya\MybbConnection\Delayers;

use Redis;
use RedisException;

/**
 * Класс для обеспечения запросов к Mybb.ru в очереди при помощи базы данных Redis
 */
class RedisDelayer extends AbstractDelayer
{
    /**
     * @var Redis Объект Redis
     */
    protected Redis $redis;

    /**
     * Redis-ключ для времени последней отправки запроса
     */
    protected string $access_key = 'mybb_access_time';

    /**
     * Redis-ключ для очереди запросов на отправку запроса
     */
    protected string $queue_key = 'mybb_access_queue';

    /**
     * Создание очереди запросов к Mybb.ru при помощи Redis
     *
     * @param string $host Хост для подключения к Redis
     * @param string|int $port Порт для подключения к Redis
     * @param string|null $password Пароль для подключения к Redis
     *
     * @throws RedisException
     */
    public function __construct(public string $host, public string|int $port, public ?string $password = null)
    {
        $this->redis = new Redis();
        $this->redis->connect($this->host, $this->port);

        if (!empty($this->password)) {
            $this->redis->auth($this->password);
        }
    }

    /**
     * Гарантированное ожидание доступа
     *
     * @throws RedisException
     */
    public function waitQueue(string $id): void
    {
        $this->redis->rPush($this->queue_key, $id);

        while ($this->redis->lIndex($this->queue_key, 0) !== $id) {
            usleep($this->min_access_delay);
        }

        self::waitAccess();

        $this->redis->lPop($this->queue_key);
    }

    /**
     * Информация о последнем запросе к серверу Mybb
     *
     * @throws RedisException
     */
    public function lastAccess(): float
    {
        return (float) $this->redis->get($this->access_key) ?? 0;
    }

    /**
     * Обновление информации о времени последнего доступа
     *
     * @throws RedisException
     */
    public function updateAccess(): void
    {
        $this->redis->set($this->access_key, microtime(true));
    }

    /**
     * Redis-ключ для времени последней отправки запроса
     *
     * @return string
     */
    public function getAccessKey(): string
    {
        return $this->access_key;
    }

    /**
     * Redis-ключ для времени последней отправки запроса
     *
     * @param string $access_key
     * @return RedisDelayer
     */
    public function setAccessKey(string $access_key): RedisDelayer
    {
        $this->access_key = $access_key;

        return $this;
    }

    /**
     * Redis-ключ для очереди запросов на отправку запроса
     *
     * @return string
     */
    public function getQueueKey(): string
    {
        return $this->queue_key;
    }

    /**
     * Redis-ключ для очереди запросов на отправку запроса
     *
     * @param string $queue_key
     * @return RedisDelayer
     */
    public function setQueueKey(string $queue_key): RedisDelayer
    {
        $this->queue_key = $queue_key;

        return $this;
    }
}