<?php

namespace Kozhilya\MybbConnection;

use CurlHandle;
use Kozhilya\MybbConnection\Exceptions\MybbApiException;
use Kozhilya\MybbConnection\Exceptions\MybbCurlException;
use Kozhilya\MybbConnection\Exceptions\MybbLoginFailedException;
use RuntimeException;

/**
 * Генератор запросов к форуму Mybb.ru
 */
final class MybbLoader
{
    public MybbConnection $connection;

    public ?MybbUser $user;

    public string $userAgent;

    /**
     * Создание объекта запросов к форуму с учётными данными определённого пользователя.
     *
     * @param MybbConnection $connection Объект для подключения к Mybb.ru
     * @param MybbUser|null $user Пользователь, подключающийся к Mybb.ru
     *
     * @throws MybbLoginFailedException Попытка входа на форум не успешна
     * @throws MybbCurlException
     */
    public function __construct(MybbConnection $connection, ?MybbUser $user = null) {
        $this->connection = $connection;
        $this->user = $user;

        $this->userAgent = 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/108.0.0.0 Safari/537.36 MyBBParser/2.1 (by Kozhilya) Curl/'.curl_version()['version'];

        if (!is_null($this->user)) {
            $this->login();
        }
    }

    /**
     * Создать Curl-объект для запроса
     *
     * @param string $url Ожидаемая ссылка запроса
     * @param array $curl_settings Дополнительные настройки Curl
     * @return CurlHandle
     */
    public function createCurl(string $url, array $curl_settings = []): CurlHandle
    {
        $url = $this->connection->buildLink($url);
        $curl = curl_init($url);

        if ($curl === false) {
            throw new RuntimeException('Ошибка при создании CURL-объекта.');
        }

        curl_setopt($curl, CURLOPT_VERBOSE, 0);
        curl_setopt($curl, CURLOPT_ENCODING, 0);
        curl_setopt($curl, CURLOPT_USERAGENT, $this->userAgent);
        curl_setopt($curl, CURLOPT_COOKIE, $this->user->getCookiesString());
        curl_setopt($curl, CURLOPT_NOPROGRESS, 1);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_HEADER, 1);
        curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 30);

        foreach ($curl_settings as $key => $value) {
            curl_setopt($curl, $key, $value);
        }

        return $curl;
    }

    /**
     * Выполнение запроса Curl
     *
     * @param CurlHandle $curl
     * @return MybbResponse
     *
     * @throws MybbCurlException
     */
    public function exec(CurlHandle $curl): MybbResponse
    {
        return new MybbResponse($this, $curl);
    }

    /**
     * Простая загрузка страницы
     *
     * @param string $url Ожидаемая ссылка запроса
     * @param array $curl_settings Дополнительные настройки Curl
     * @return MybbResponse
     *
     * @throws MybbCurlException
     */
    public function load(string $url, array $curl_settings = []): MybbResponse
    {
        $curl = $this->createCurl($url, $curl_settings);

        return $this->exec($curl);
    }

    /**
     * API-запрос к MyBB-серверу
     *
     * @param string $method Метод API
     * @param array|null $params Параметры API-запроса
     *
     * @return mixed
     *
     * @throws MybbApiException
     * @throws MybbCurlException
     *
     * @see https://mybb.ru/forumapi/
     */
    public function api(string $method, ?array $params = null): mixed
    {
        $params = $params ?? [];
        $url = $this->connection->buildLink(sprintf(
            "api.php?method=%s&%s",
            urlencode($method),
            http_build_query($params)
        ));

        $handle = $this->createCurl($url);
        $response = new MybbResponse($this, $handle);

        $responseJson = $response->json();

        if (isset($responseJson['response']['error'])) {
            throw new MybbApiException($responseJson['response']['error'], $method, $params);
        }

        return $responseJson;
    }

    /**
     * Попытка авторизации на форум
     *
     * @return void
     * @throws MybbLoginFailedException
     * @throws MybbCurlException
     */
    public function login(): void
    {
        try {
            $json = $this->api('board.auth', [
                'login' => $this->user->getLogin(),
                'password' => $this->user->getPassword(),
            ]); //->json();

            $this->user->addCookies([
                'mybb_ru' => $json['response']['hash']
            ]);
        }
        catch (MybbApiException $apiError) {
            throw new MybbLoginFailedException($this, $apiError->getMybbMessage());
        }
    }
}