<?php

namespace Kozhilya\MybbConnection;

use CurlHandle;
use Kozhilya\MybbConnection\Exceptions\MybbCurlException;
use Kozhilya\MybbConnection\Util\MybbEncoder;

/**
 * Ответ на запрос от форума Mybb.ru
 */
final class MybbResponse
{
    /**
     * ID текущего сообщения
     * @var string
     */
    private string $id;

    /**
     * Загрузчик, создавший запрос
     * @var MybbLoader
     */
    private MybbLoader $parent;

    /**
     * Curl-объект запроса
     * @var CurlHandle
     */
    private CurlHandle $handle;

    /**
     * Ответ на запрос
     * @var mixed|array|bool|string|string[]
     */
    public mixed $response;

    /**
     * Заголовочная часть запроса
     * @var string
     */
    public string $header;

    /**
     * Содержательная часть запроса
     * @var string
     */
    public string $html;

    /**
     * Создание нового запроса
     *
     * @param MybbLoader $parent Загрузчик, создающий запрос
     * @param CurlHandle $handle CURL-объект запроса
     *
     * @throws MybbCurlException Ошибка CURL
     */
    public function __construct(MybbLoader $parent, CurlHandle $handle)
    {
        $this->id = uniqid('mybb_request', true);
        $this->parent = $parent;
        $this->handle = $handle;

        $delayer = $this->parent->connection->getDelayer();

        $delayer->waitQueue($this->id);
        $this->response = curl_exec($this->handle);
        $delayer->updateAccess();

        if ($this->response === false) {
            $err = curl_error($this->handle);
            curl_close($this->handle);

            throw new MybbCurlException($this, $err);
        }

        curl_close($this->handle);

        $this->response = str_replace("\r", '', $this->response);

        $this->header = substr($this->response, 0, strpos($this->response, "\n\n"));

        $this->parent->user->addCookies($this->getCookies());

        $this->html = substr($this->response, strpos($this->response, "\n\n") + 2);

        $encoding = $this->getReportedEncoding();

        if (mb_strtolower($encoding) != 'utf-8') {
            $this->html = MybbEncoder::fromForumEncoding($this->html);
        }
    }

    /**
     * Получение использованной ссылки на запрос
     *
     * @return string
     */
    public function getUrl(): string {
        return curl_getinfo($this->handle, CURLINFO_EFFECTIVE_URL);
    }

    /**
     * Получение cookies, отправленных сервером
     *
     * @return array Список новых cookie
     */
    private function getCookies(): array
    {
        preg_match_all('#^Set-Cookie:\s*(.*)$#mi', $this->header, $matches);
        $cookies = [];

        foreach ($matches[1] as $raw_cookie) {
            parse_str(strtr($raw_cookie, array('&' => '%26', '+' => '%2B', ';' => '&')), $part);

            $cookies = array_merge($cookies, $part);
        }

        unset($cookies['expires']);
        unset($cookies['path']);

        return $cookies;
    }

    /**
     * Получение кодировки, заявленной сервером
     *
     * @return string
     */
    private function getReportedEncoding(): string
    {
        preg_match('#charset\s*=\s*(.*)$#mi', $this->header, $match);

        return $match[1];
    }

    /**
     * Получение JSON-содержимого
     *
     * @return mixed
     */
    public function json(): mixed
    {
        return json_decode($this->html ?? '{}');
    }

}